/* 
 * log.c 
 *
 * Створює задану кількість (THREADS_NUM) потоків, кожен із яких 
 * відкриває власний журнальний файл і записує в нього повідомлення, 
 * після чого завершує роботу. Закриття журнального файлу виконується
 * через обробник очистки.
 * Ілюструє порядок застосування обробників очистки (функції
 * pthread_cleanup_push(), pthread_cleanup_pop()).
 *
 */

#include <assert.h>
#include <pthread.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* Кількість дочірніх потоків */
enum { THREADS_NUM = 5 };

/* Масив покажчиків на журнальні файли */
FILE *thread_logs[THREADS_NUM];

void write_to_thread_log(unsigned int thread_num, const char *message);
void close_thread_log(void* thread_log);
void *thread_function(void *arg);


/* Записує повідомлення message в журнальний файл із номером
   thread_num. */
void write_to_thread_log(unsigned int thread_num, const char *message)
{
        assert(thread_num < THREADS_NUM);

        if (thread_logs[thread_num] != NULL)
                fputs(message, thread_logs[thread_num]);
}

/* Обробник очистки.
   Закриває журнальний файл, на який указує thread_log. */
void close_thread_log(void* thread_log)
{
        if (thread_log != NULL)
                fclose((FILE *) thread_log);
}

/* Потокова функція */
void *thread_function(void *arg)
{
        enum { LOGNAME_LEN = 20 };
        char logname[LOGNAME_LEN];
        unsigned int thread_num = *((unsigned int *) arg);

        assert(thread_num < THREADS_NUM);
        /* Створює приватне ім'я журнального файлу. */
        snprintf(logname, LOGNAME_LEN, "thread%u.log", thread_num);
        /* Відкриває журнальний файл. */
        thread_logs[thread_num] = fopen(logname, "w");
        if (thread_logs[thread_num] == NULL)
                fprintf(stderr, "Error opening thread log file %s\n", logname);

        /* Реєструє обробник очистки. */
        pthread_cleanup_push(close_thread_log,
                                        (void *) thread_logs[thread_num]);

        /* Записує початкове повідомлення в журнальний файл. */
        write_to_thread_log(thread_num, "Thread has started.\n");

        /* Далі повинна йти основна частина тіла потокової функції... */

        /* Виконує і знімає з реєстрації обробник очистки. */
        /* pthread_cleanup_push() повинна йти в парі з
           pthread_cleanup_pop(), причому на тому самому рівні
           вкладеності. */
        pthread_cleanup_pop(1);
        return NULL;
}

int main()
{
        pthread_t threads[THREADS_NUM];
        unsigned int thread_args[THREADS_NUM]; 
        unsigned int i;
        int terrno;

        /* Створює потоки. */
        for (i = 0; i < THREADS_NUM; i++) {
                thread_args[i] = i;
                terrno = pthread_create(&(threads[i]), NULL,
                                thread_function, (void *) &thread_args[i]);
                if (terrno != 0) {
                        fprintf(stderr, "Error creating thread: %s\n",
                                                        strerror(terrno));
                        exit(EXIT_FAILURE);
                }
        }

        /* Чекає на завершення всіх потоків. */
        for (i = 0; i < THREADS_NUM; i++) {
                terrno = pthread_join(threads[i], NULL);
                if (terrno != 0) {
                        fprintf(stderr, "Error joining thread: %s\n",
                                                        strerror(terrno));
                        exit(EXIT_FAILURE);
                }
        }

        return EXIT_SUCCESS;
}
